package com.bakinsbits.example;

import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.LinkedBlockingQueue;

import com.bakinsbits.example.Table;

public class TableImpl implements Table {

	final Deck deck;
	private final Set<PlayerWrapper> players;
	private final Thread pusher;
	/* The actionQueue contains either PlayerWrappers - meaning, send that one
	 * player the current deck, or a TableImpl (or anything else, really) - meaning
	 * shuffle and then send all players the new deck.
	 */
	private final LinkedBlockingQueue<Object> actionQueue;

	public TableImpl() throws RemoteException {
		super();		
		System.out.println("TableImpl: constructor");
		deck = new Deck();
		players = Collections.synchronizedSet(new HashSet<PlayerWrapper>());
		actionQueue = new LinkedBlockingQueue<Object>();
		pusher = new Thread(new DeckPusher());
		pusher.setDaemon(true);
		pusher.start();
	}

	@Override
	public Dealer joinTable(String name, Player player) throws RemoteException {
		System.out.println(String.format("TableImpl: joinTable(%s,%s)", name, player));
		PlayerWrapper pl = new PlayerWrapper(name, player);
		players.add(pl);
		DealerImpl dealer = new DealerImpl();
		dealer.setTable(this);
		dealer.setPlayer(pl);
		UnicastRemoteObject.exportObject(dealer, 0);
		return dealer;
	}
	
	void leaveTable(PlayerWrapper player) {
		System.out.println(String.format("TableImpl: leaveTable(%s)", player));
		players.remove(player);
	}
	
	void getDeck(PlayerWrapper player) {
		System.out.println(String.format("TableImpl: getDeck(%s) enqueuing", player));
		actionQueue.offer(player);
	}
	
	void shuffle() {
		System.out.println(String.format("TableImpl: shuffle() enqueuing"));
		actionQueue.offer(this);
	}
	
	private class DeckPusher implements Runnable {
	
		@Override
		public void run() {		
			Set<PlayerWrapper> deadPlayers = new HashSet<PlayerWrapper>();
			int[] cards = deck.getCards();
			Object o;
			while (true) {
				System.out.println("DeckPusher: taking from queue");
				try {
					o = actionQueue.take();
				} catch (InterruptedException e1) {
					return;
				}
				System.out.println(String.format("DeckPusher: took %s from queue", o));
				if (o instanceof PlayerWrapper) {
					PlayerWrapper player = (PlayerWrapper)o;
					System.out.println(String.format("DeckPusher: sending %s the cards", player));
					try {
						player.deck(cards);
					} catch (RemoteException e) {
						System.out.println(String.format("dead player %s (%s)", player, e));
					}
				}
				else {
					System.out.println(String.format("DeckPusher: shuffling then sending cards to %d players", players.size()));
					deck.shuffle();
					cards = deck.getCards();
					PlayerWrapper[] playersAtTable = players.toArray(new PlayerWrapper[0]);
					for (PlayerWrapper player : playersAtTable) {
						try {
							player.deck(cards);
						} catch (RemoteException e) {
							// abandon this player
							System.out.println(String.format("dead player %s (%s)", player, e));
							deadPlayers.add(player);
						}
					}
					for (PlayerWrapper player : deadPlayers) {
						players.remove(player);
					}
				}
			}
		}
	}

	@Override
	public void ping() {
		System.out.println("Table: Ping!");
	}
}
