package com.bakinsbits.spigot;

// Per Gibbons' article on the unbounded spigot algorithm for pi, pg. 5

public class StreamingAlgorithm {
	
	// Initially described in terms of infinite input and output lists, this
	// version takes objects which source or sink an element
	// A: input type
	// B: state type
	// C: output type
	
    // Will terminate either on end of input, or when a (user-supplied) terminate 
	// function on the state returns true
	
	public interface NextInput<A> {
		public boolean hasNextInput();
		public A nextInput();
	}
	
	public interface NextOutput<C> {
		public void nextOutput(C c);
	}
	
	public interface Streamer<A,B,C> {
		public B cons(B b, A a);
		public C next(B b);
		public B prod(B b, C c);
		public boolean safe(B b, C c);
		public boolean terminate(B b); // termination function (not in original algorithm)
	}
	

	// transcription without using tail call
	public <A, B, C> void stream(
			Streamer<A,B,C> streamer,
			B z,
			NextInput<A> input, 
			NextOutput<C> output) {
		
		while (!streamer.terminate(z)) {
			C y = streamer.next(z);			
			if (streamer.safe(z, y)) {
				z = streamer.prod(z, y);
                output.nextOutput(y);
			} else if (!input.hasNextInput()) {
				break;
			} else {
				z = streamer.cons(z, input.nextInput());
			}
		}
	}
	
	// transcription using tail calls
	public <A, B, C> void streamTailCall(
			Streamer<A,B,C> streamer,
			B z,
			NextInput<A> input, 
			NextOutput<C> output) {
		
		if (streamer.terminate(z))
			return;
		
		C y = streamer.next(z);
		
		if (streamer.safe(z, y)) {
			output.nextOutput(y);
			streamTailCall(streamer, streamer.prod(z, y), input, output); // tail call
		} else {
			if (!input.hasNextInput()) {
				// terminate at end of input - not quite right since you might
				// want to process the remaining state
				return;
			}
			A x = input.nextInput();
			streamTailCall(streamer, streamer.cons(z, x), input, output); // tail call
		}		
	}

}
