package com.bakinsbits.spigot;

import java.math.BigInteger;

/**
 * PiSpigotLFT = linear functional transformation specialized for the PiSpigot
 * algorithm, where the lower-left element (s) is always zero! 
 */
public class PiSpigotLFT implements LinearFunctionalTransformation {
	
	public static BigInteger ONE  = BigInteger.ONE;
	public static BigInteger ZERO = BigInteger.ZERO;
	
	public static LinearFunctionalTransformation UNIT = new PiSpigotLFT(ONE, ZERO, ZERO, ONE); 
	
	private BigInteger q, r, t;
	// Forming the matrix:
	//  ( q r )
	//  ( s t )
	
	PiSpigotLFT(BigInteger q, BigInteger r, BigInteger s, BigInteger t)
	{
		// Must have s == 0
		if (!s.equals(ZERO)) {
			throw new IllegalArgumentException("s must be 0");
		}
		
		// Must have qt - rs != 0
		if (q.multiply(t).equals(ZERO)) {
			throw new IllegalArgumentException();
		}
		
		this.q = q;
		this.r = r;
		this.t = t;
	}
	
	/* (non-Javadoc)
	 * @see com.bakinsbits.spigot.LinearFunctionalTransformation#comp(com.bakinsbits.spigot.LFT)
	 */
	@Override
	public LinearFunctionalTransformation comp(LinearFunctionalTransformation b)
	{
		if (! (b instanceof PiSpigotLFT)) {
			throw new IllegalArgumentException("PiSpigotLFT only compatible with other PiSpigotLFTs");
		}
		
		BigInteger q = this.q.multiply(b.getQ());
		BigInteger r = this.q.multiply(b.getR()).add(this.r.multiply(b.getT()));
	    BigInteger t = this.t.multiply(b.getT());
		
	    // Reduce at each LFT step.   More sophisticated might be to reduce less
	    // frequently, perhaps looking at the bitlengths of the integers.
	    BigInteger gcd = q.gcd(r);
	    if (!gcd.equals(ONE)) {
	    	gcd = gcd.gcd(t);
	    }
	    
	    if (gcd.equals(ONE)) {
	    	return new PiSpigotLFT(q,r,ZERO,t);
	    } else {
	    	return new PiSpigotLFT(q.divide(gcd),r.divide(gcd),ZERO,t.divide(gcd));
	    }
	    
	}
	
	/* (non-Javadoc)
	 * @see com.bakinsbits.spigot.LinearFunctionalTransformation#extr(java.math.BigInteger)
	 */
	@Override
	public BigRational extr(BigInteger x)
	{
		return new BigRational(q.multiply(x).add(r), t);
	}

    @Override
    public BigRational extr(BigRational x)
    {
        BigRational num = new BigRational(q).multiply(x).add(new BigRational(r));
        BigRational den = new BigRational(t);
        return num.divide(den);
    }

	@Override
	public BigInteger getQ() {
		return q;
	}

	@Override
	public BigInteger getR() {
		return r;
	}

	@Override
	public BigInteger getS() {
		return ZERO;
	}

	@Override
	public BigInteger getT() {
		return t;
	}
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("(");
        sb.append(getQ().toString());
        sb.append(",");
        sb.append(getR().toString());
        sb.append(",");
        sb.append(getS().toString());
        sb.append(",");
        sb.append(getT().toString());
        sb.append(")");
        return sb.toString();
    }
}
