package com.bakinsbits.spigot;

import static java.lang.System.out;
import java.math.BigInteger;

/**
 * Unbounded spigot algorithm for pi (following Gibbons' paper)
 * <p>
 * Uses the StreamingAlgorithm class, but this class implements itself all the
 * interfaces required by the streaming algorithm.
 */
public class PiSpigot implements 
						StreamingAlgorithm.NextInput<LinearFunctionalTransformation>,
						StreamingAlgorithm.NextOutput<Integer>,
						StreamingAlgorithm.Streamer<
							LinearFunctionalTransformation, 
							LinearFunctionalTransformation, 
							Integer> {
	
	// A = input  = PiSpigotLFT
	// B = state  = PiSpigotLFT
	// C = output = Integer
	
	// TODO: Read in pi digits base 10 from file and compare with output (collected in a list)
	// TODO: Compose with BaseConversion to get pi in hex, and compare that with a file too
	
	private int k = 0;  // next sequence element for input
	
	@Override
	public boolean hasNextInput() {
		// infinite list
		return true;
	}
	
	@Override
	public LinearFunctionalTransformation nextInput() {
		k++;
		return new PiSpigotLFT(
				BigInteger.valueOf(k), 
				BigInteger.valueOf(4L*k+2),
				BigInteger.ZERO, 
				BigInteger.valueOf(2L*k+1)
				);
	}
	
	static final BigInteger FOUR  = BigInteger.valueOf(4);
	static final BigInteger ONE   = BigInteger.ONE;
	static final BigInteger TEN   = BigInteger.TEN;
	static final BigInteger THREE = BigInteger.valueOf(3);
	static final BigInteger ZERO  = BigInteger.ZERO;

	@Override
	public LinearFunctionalTransformation cons(LinearFunctionalTransformation b, 
	                                           LinearFunctionalTransformation c) {
		return b.comp(c);
	}

	@Override
	public Integer next(LinearFunctionalTransformation b) {
		return b.extr(THREE).floor().intValue();
	}

	@Override
	public LinearFunctionalTransformation prod(LinearFunctionalTransformation b,
	                                           Integer c) {
		return new PiSpigotLFT(
				TEN, 
				BigInteger.valueOf(-10 * c),
				ZERO,
				ONE).comp(b);
	}

	@Override
	public boolean safe(LinearFunctionalTransformation b, Integer c) {
		return c == b.extr(FOUR).floor().intValue();
	}

	@Override
	public boolean terminate(LinearFunctionalTransformation b) {
		// TODO: Could terminate based on outputDigit (if this class wasn't static)
		return false;
	}
		
	private int outputDigitPlace = -1;	// next output digit place, for counting the digits output
	
	@Override
	public void nextOutput(Integer i) {
		out.format("%1d", i);
		outputDigitPlace++;
		if (outputDigitPlace == 0)
			out.print(".");
		else if (outputDigitPlace % 20 == 0)
			out.format(" - %d%n  ", outputDigitPlace);
	}
	
	public void stream() {
		new StreamingAlgorithm().stream(this, PiSpigotLFT.UNIT, this, this);
	}

	public static void main(String... args) {
		new PiSpigot().stream();
	}
}
