package com.bakinsbits.spigot;

import static java.lang.System.out;
import java.math.BigInteger;

/**
 * Unbounded spigot algorithm for pi (following Gibbons' paper) using Lambert's
 * expression.
 * <p>
 * Uses the StreamingAlgorithm class, but this class implements itself all the
 * interfaces required by the streaming algorithm.
 */

final class LambertState {
	public LinearFunctionalTransformation lft;
	public int consumed;
	LambertState(LinearFunctionalTransformation lft, int consumed) {
		this.lft = lft;
		this.consumed = consumed;
	}
}

public class LambertPiSpigot implements 
						StreamingAlgorithm.NextInput<LinearFunctionalTransformation>,
						StreamingAlgorithm.NextOutput<Integer>,
						StreamingAlgorithm.Streamer<
							LinearFunctionalTransformation, 
							LambertState, 
							Integer> {
	
	// A = input  = LFT
	// B = state  = (LFT, number-of-terms-consumed)
	// C = output = Integer
	
	// TODO: Read in pi digits base 10 from file and compare with output (collected in a list)
	// TODO: Compose with BaseConversion to get pi in hex, and compare that with a file too
	
	private int k = 0;  // next sequence element for input
	
	@Override
	public boolean hasNextInput() {
		// infinite list
		return true;
	}
	
	@Override
	public LinearFunctionalTransformation nextInput() {
		k++;
		return new LFT(
				BigInteger.valueOf(2L*k-1), 
				BigInteger.valueOf((long)k*k),
				ONE, 
				ZERO);
	}
	
	static final BigInteger FOUR  = BigInteger.valueOf(4);
	static final BigInteger MTEN  = BigInteger.valueOf(-10);
	static final BigInteger ONE   = BigInteger.ONE;
	static final BigInteger TEN   = BigInteger.TEN;
	static final BigInteger THREE = BigInteger.valueOf(3);
	static final BigInteger TWO   = BigInteger.valueOf(2);
	static final BigInteger ZERO  = BigInteger.ZERO;

	@Override
	public LambertState cons(LambertState b, LinearFunctionalTransformation c) {
		return new LambertState(b.lft.comp(c), b.consumed + 1);
	}

	@Override
	public Integer next(LambertState b) {
		BigInteger x = BigInteger.valueOf(2 * b.consumed - 1);
		return new BigRational(
				b.lft.getQ().multiply(x).add(b.lft.getR()),
				b.lft.getS().multiply(x).add(b.lft.getT()))
			.floor().intValue();
	}

	@Override
	public LambertState prod(LambertState b, Integer c) {
		return new LambertState(
		   new LFT(
				TEN, 
				MTEN.multiply(BigInteger.valueOf(c)), 
				ZERO,
				ONE).comp(b.lft), b.consumed);
	}

	@Override
	public boolean safe(LambertState b, Integer c) {
		BigInteger x = BigInteger.valueOf(5L * b.consumed - 2);
		return c == new BigRational(
							b.lft.getQ().multiply(x).add(b.lft.getR().multiply(TWO)),
							b.lft.getS().multiply(x).add(b.lft.getT().multiply(TWO)))
					.floor().intValue();
	}

	@Override
	public boolean terminate(LambertState b) {
		// TODO: Could terminate based on outputDigit (if this class wasn't static)
		return false;
	}
		
	private int outputDigitPlace = -1;	// next output digit place, for counting the digits output
	
	@Override
	public void nextOutput(Integer i) {
		out.format("%1d", i);
		outputDigitPlace++;
		if (outputDigitPlace == 0)
			out.print(".");
		else if (outputDigitPlace % 20 == 0)
			out.format(" - %d%n  ", outputDigitPlace);
	}
	
	public void stream() {
		LFT initialLFT = new LFT(ZERO, FOUR, ONE, ZERO);
		LambertState initialState = new LambertState(initialLFT, 1);
		new StreamingAlgorithm().stream(this, initialState, this, this);
	}

	public static void main(String... args) {
		new LambertPiSpigot().stream();
	}
}
