package com.bakinsbits.spigot;

import java.math.BigInteger;

/**
 * LFT = linear functional transformation
 */
public class LFT implements LinearFunctionalTransformation {
	
	public static BigInteger ONE  = BigInteger.ONE;
	public static BigInteger ZERO = BigInteger.ZERO;
	
	public static LinearFunctionalTransformation UNIT = new LFT(ONE, ZERO, ZERO, ONE); 
	
	private BigInteger q, r, s, t;
	// Forming the matrix:
	//  ( q r )
	//  ( s t )
	
	LFT(BigInteger q, BigInteger r, BigInteger s, BigInteger t)
	{
		// Must have qt - rs != 0
		if (q.multiply(t).subtract(r.multiply(s)).equals(ZERO))
			throw new IllegalArgumentException();
		
		this.q = q;
		this.r = r;
		this.s = s;
		this.t = t;
	}
	
	/* (non-Javadoc)
	 * @see com.bakinsbits.spigot.LinearFunctionalTransformation#comp(com.bakinsbits.spigot.LFT)
	 */
	@Override
	public LinearFunctionalTransformation comp(LinearFunctionalTransformation b)
	{
		BigInteger q = this.q.multiply(b.getQ()).add(this.r.multiply(b.getS()));
		BigInteger r = this.q.multiply(b.getR()).add(this.r.multiply(b.getT()));
	    BigInteger s = this.s.multiply(b.getQ()).add(this.t.multiply(b.getS()));
	    BigInteger t = this.s.multiply(b.getR()).add(this.t.multiply(b.getT()));
		
	    // Reduce at each LFT step.   More sophisticated might be to reduce less
	    // frequently, perhaps looking at the bitlengths of the integers.
	    BigInteger gcd = q.gcd(r);
	    if (!gcd.equals(ONE)) {
	    	gcd = gcd.gcd(s);
	    }
	    if (!gcd.equals(ONE)) {
	    	gcd = gcd.gcd(t);
	    }
	    
	    if (gcd.equals(ONE)) {
	    	return new LFT(q,r,s,t);
	    } else {
	    	return new LFT(q.divide(gcd),r.divide(gcd),s.divide(gcd),t.divide(gcd));
	    }
	    
	}
	
	/* (non-Javadoc)
	 * @see com.bakinsbits.spigot.LinearFunctionalTransformation#extr(java.math.BigInteger)
	 */
	@Override
	public BigRational extr(BigInteger x)
	{
		return new BigRational(x.multiply(q).add(r), x.multiply(s).add(t));
	}
	
	@Override
	public BigRational extr(BigRational x)
	{
	    BigRational num = x.multiply(new BigRational(q)).add(new BigRational(r));
	    BigRational den = x.multiply(new BigRational(s)).add(new BigRational(t));
	    return num.divide(den);
	}

	@Override
	public BigInteger getQ() {
		return q;
	}

	@Override
	public BigInteger getR() {
		return r;
	}

	@Override
	public BigInteger getS() {
		return s;
	}

	@Override
	public BigInteger getT() {
		return t;
	}
	
	@Override
	public String toString() {
	    StringBuilder sb = new StringBuilder();
	    sb.append("(");
	    sb.append(getQ().toString());
	    sb.append(",");
	    sb.append(getR().toString());
	    sb.append(",");
	    sb.append(getS().toString());
	    sb.append(",");
	    sb.append(getT().toString());
	    sb.append(")");
	    return sb.toString();
	}
}
