package com.bakinsbits.spigot;

import static java.lang.System.out;
import java.math.BigInteger;
import java.util.List;
import java.util.ArrayList;

/**
 * Unbounded spigot algorithm for pi (following Gibbons' paper) using Gosper's
 * formula.
 * <p>
 * Uses the StreamingAlgorithm class, but this class implements itself all the
 * interfaces required by the streaming algorithm.
 */

public class GosperPiSpigotAnnotated implements 
						StreamingAlgorithm.NextInput<LinearFunctionalTransformation>,
						StreamingAlgorithm.NextOutput<Integer>,
						StreamingAlgorithm.Streamer<
							LinearFunctionalTransformation, 
							GosperState, 
							Integer> {
	
	// A = input  = PiSpigotLFT
	// B = state  = (PiSpigotLFT, number-of-terms-consumed)
	// C = output = Integer
	
	private int i = 0;  // next sequence element for input
	
	@Override
	public boolean hasNextInput() {
		// infinite list
		return true;
	}
	
	@Override
	public LinearFunctionalTransformation nextInput() {
	    // Note: After only 275+ terms the value of r overflows an int
		i++;
		return new PiSpigotLFT(
		        BigInteger.valueOf(i*(2L*i-1)),
		        BigInteger.valueOf(3L*(3*i+1)*(3*i+2)*(5*i-2)),
		        ZERO,
		        BigInteger.valueOf(3L*(3*i+1)*(3*i+2))
		        );
	}

	static final BigInteger FIVE  = BigInteger.valueOf(5);
	static final BigInteger ONE   = BigInteger.ONE;
	static final BigInteger ONEHUNDREDTWENTYFIVE = BigInteger.valueOf(125);
	static final BigInteger TEN   = BigInteger.TEN;
	static final BigInteger ZERO  = BigInteger.ZERO;
	
	@Override
	public GosperState cons(GosperState b, LinearFunctionalTransformation z_) {
//		return new GosperState(b.z.comp(z_), b.i+1);
	    
	    LinearFunctionalTransformation zc = b.z.comp(z_);
	    
	    appendLog("cons i", String.valueOf(b.i));
        appendLog("cons z", b.z.toString());
        appendLog("cons z'", z_.toString());
	    appendLog("cons zc", zc.toString());
	    
	    return new GosperState(zc, b.i+1);
	}

	@Override
	public Integer next(GosperState b) {
//	    BigRational v = new BigRational(BigInteger.valueOf(27L*b.i - 12), FIVE);
//	    return b.z.extr(v).floor().intValue();

	    BigRational v = new BigRational(BigInteger.valueOf(27L*b.i - 12), FIVE);
	    BigRational e = b.z.extr(v);
	    BigInteger f = e.floor();
	    
	    appendLog("next i", String.valueOf(b.i));
	    appendLog("next v", v.toString());
	    appendLog("next e", e.toString());
	    appendLog("next f", f.toString());
	    
	    return f.intValue();	    
	}

	@Override
	public GosperState prod(GosperState b, Integer n) {
//		return new GosperState(
//				new PiSpigotLFT(
//						TEN, 
//						BigInteger.valueOf(-10 * n),
//						ZERO, 
//						ONE).comp(b.z), 
//				b.i);

	    LinearFunctionalTransformation zp = new PiSpigotLFT(
                        TEN, 
                        BigInteger.valueOf(-10 * n),
                        ZERO, 
                        ONE);
	    LinearFunctionalTransformation zc = zp.comp(b.z);
	    
	    appendLog("prod i", String.valueOf(b.i));
	    appendLog("prod n", n.toString());
	    appendLog("prod zp", zp.toString());
        appendLog("prod z", b.z.toString());
	    appendLog("prod zc", zc.toString());
	    
	    return new GosperState(zc, b.i);
	}

	@Override
	public boolean safe(GosperState b, Integer n) {
	    // TODO: There's something wrong with this calculation of "safe":
	    
//	    BigRational v = new BigRational(BigInteger.valueOf(675L*b.i - 216), 
//	                                    ONEHUNDREDTWENTYFIVE);
//	    return b.z.extr(v).floor().intValue() == n;
	    
        BigRational v = new BigRational(BigInteger.valueOf(675L*b.i - 216), 
                                        ONEHUNDREDTWENTYFIVE);
        BigRational e = b.z.extr(v);
        BigInteger f = e.floor();
        boolean res = n == f.intValue();
        
        appendLog("safe i", String.valueOf(b.i));
        appendLog("safe v", v.toString());
        appendLog("safe e", e.toString());
        appendLog("safe f", f.toString());
        appendLog("safe n", n.toString());
        appendLog("safe res", res ? "True" : "False");
        
        return b.z.extr(v).floor().intValue() == n;
	}

	@Override
	public boolean terminate(GosperState b) {
	    return outputDigitPlace >= limit;
	}
		
	// Next output digit place (for counting the digits output, for formatting)
	private int outputDigitPlace = -1;
	
	@Override
	public void nextOutput(Integer i) {
//	    if (i < 0 || i > 9) {
//	        out.format("(%d)", i);
//	    } else {
//	        out.format("%1d", i);
//	    }
//		outputDigitPlace++;
//		if (outputDigitPlace == 0)
//			out.print(".");
//		else if (outputDigitPlace % 20 == 0)
//			out.format(" - %d%n  ", outputDigitPlace);
	    
	    outputDigitPlace++;
	    outLog(i);
	}
	
	private int limit = 999999999;
	public void stream(int limit) {
	    this.limit = limit;
	    out.format("[%n");
		new StreamingAlgorithm().stream(
		        this, 
		        new GosperState(PiSpigotLFT.UNIT, 1), 
		        this, 
		        this);
		out.format("]%n");
	}
	
	private class LogEntry {
	    final private String descr;
	    final private String value;
	    LogEntry(String descr, String value) {
	        this.descr = descr;
	        this.value = value;
	    }
	    public String toString() {
	        StringBuilder sb = new StringBuilder();
	        sb.append("(\"");
	        sb.append(descr);
	        sb.append("\",\"");
	        sb.append(value);
	        sb.append("\")");
	        return sb.toString();
	    }
	}
	
	private List<LogEntry> log = new ArrayList<LogEntry>();
	
	private void clearLog() {
	    log = new ArrayList<LogEntry>();
	}
	
	private void appendLog(String descr, String value) {
	    log.add(new LogEntry(descr, value));
	}
	
	private void outLog(Integer i) {
	    // dump the integer i and the corresponding log.  also, clear the log
	    out.format("(%d,[", i);
	    boolean first = true;
	    for (LogEntry e : log ) {
	        if (first) {
	            first = false;
	        } else {
	            out.format(",");
	        }
	        out.format("%s", e.toString());
	    }
	    out.format("]),%n"); // extra comma after very last digit's log dump
	    clearLog();
	}
	
	// N.B.: This is piG3 from the paper.
	public void fastStream() {
	    BigInteger q = ONE;
	    BigInteger r = BigInteger.valueOf(180);
	    BigInteger t = BigInteger.valueOf(60);
	    int i = 2;
	    int outputDigitPlace = -1;
	    
	    while (true) {
	        long u = 3L * (3*i+1) * (3*i+2);
	        BigInteger yNum = q.multiply(BigInteger.valueOf(27L*i-12))
	                                .add(r.multiply(FIVE));
	        BigInteger yDen = t.multiply(FIVE);
	        BigInteger y = new BigRational(yNum, yDen).floor();
	        
	        if (y.signum()<0 || y.compareTo(TEN)>=0) {
	            out.format("(%s)", y.toString());
	        } else {
	            out.format("%1s", y.toString());
	        }
	        outputDigitPlace++;
	        if (outputDigitPlace == 0)
	            out.print(".");
	        else if (outputDigitPlace % 20 == 0)
	            out.format(" - %d%n  ", outputDigitPlace);
	        
	        BigInteger qp = q.multiply(BigInteger.valueOf(10L*i*(2*i-1)));
	        BigInteger rp = q.multiply(BigInteger.valueOf(5L*i-2))
	                            .add(r)
	                            .subtract(y.multiply(t))
	                            .multiply(BigInteger.valueOf(10*u));
	        BigInteger tp = t.multiply(BigInteger.valueOf(u));
	        
	        q = qp;
	        r = rp;
	        t = tp;
	        i++;
	    }
	}

	public static void main(String... args) {
	    new GosperPiSpigotAnnotated().stream(300);
//		new GosperPiSpigot().fastStream();
	}
}
