package com.bakinsbits.spigot;

import static java.lang.System.out;
import java.math.BigInteger;

/**
 * Unbounded spigot algorithm for pi (following Gibbons' paper) using Gosper's
 * formula.
 * <p>
 * Uses the StreamingAlgorithm class, but this class implements itself all the
 * interfaces required by the streaming algorithm.
 */

final class GosperState {
	final public LinearFunctionalTransformation z;
	final public int i;
	GosperState(LinearFunctionalTransformation z, int i) {
		this.z = z;
		this.i = i;
	}
}


public class GosperPiSpigot implements 
						StreamingAlgorithm.NextInput<LinearFunctionalTransformation>,
						StreamingAlgorithm.NextOutput<Integer>,
						StreamingAlgorithm.Streamer<
							LinearFunctionalTransformation, 
							GosperState, 
							Integer> {
	
	// A = input  = PiSpigotLFT
	// B = state  = (PiSpigotLFT, number-of-terms-consumed)
	// C = output = Integer
	
	private int i = 0;  // next sequence element for input
	
	@Override
	public boolean hasNextInput() {
		// infinite list
		return true;
	}
	
	@Override
	public LinearFunctionalTransformation nextInput() {
		i++;
		return new PiSpigotLFT(
		        BigInteger.valueOf(i*(2L*i-1)),
		        BigInteger.valueOf(3L*(3*i+1)*(3*i+2)*(5*i-2)),
		        ZERO,
		        BigInteger.valueOf(3L*(3*i+1)*(3*i+2))
		        );
	}

	static final BigInteger FIVE  = BigInteger.valueOf(5);
	static final BigInteger ONE   = BigInteger.ONE;
	static final BigInteger ONEHUNDREDTWENTYFIVE = BigInteger.valueOf(125);
	static final BigInteger TEN   = BigInteger.TEN;
	static final BigInteger ZERO  = BigInteger.ZERO;
	
	@Override
	public GosperState cons(GosperState b, LinearFunctionalTransformation z_) {
		return new GosperState(b.z.comp(z_), b.i+1);
	}

	@Override
	public Integer next(GosperState b) {
	    BigRational v = new BigRational(BigInteger.valueOf(27L*b.i - 12), FIVE);
	    return b.z.extr(v).floor().intValue();
	}

	@Override
	public GosperState prod(GosperState b, Integer n) {
		return new GosperState(
				new PiSpigotLFT(
						TEN, 
						BigInteger.valueOf(-10 * n),
						ZERO, 
						ONE).comp(b.z), 
				b.i);
	}

	@Override
	public boolean safe(GosperState b, Integer n) {
	    BigRational v = new BigRational(BigInteger.valueOf(675L*b.i - 216), 
	                                    ONEHUNDREDTWENTYFIVE);
	    return b.z.extr(v).floor().intValue() == n;
	}

	@Override
	public boolean terminate(GosperState b) {
		// TODO: Could terminate based on outputDigitPlace
		return false;
	}
		
	// Next output digit place (for counting the digits output, for formatting)
	private int outputDigitPlace = -1;
	
	@Override
	public void nextOutput(Integer i) {
	    if (i < 0 || i > 9) {
	        out.format("(%d)", i);
	    } else {
	        out.format("%1d", i);
	    }
		outputDigitPlace++;
		if (outputDigitPlace == 0)
			out.print(".");
		else if (outputDigitPlace % 20 == 0)
			out.format(" - %d%n  ", outputDigitPlace);
	}
	
	public void stream() {
		new StreamingAlgorithm().stream(
		        this, 
		        new GosperState(PiSpigotLFT.UNIT, 1), 
		        this, 
		        this);
	}
	
	// N.B.: This is piG3 from the paper.
	public void fastStream() {
	    BigInteger q = ONE;
	    BigInteger r = BigInteger.valueOf(180);
	    BigInteger t = BigInteger.valueOf(60);
	    int i = 2;
	    int outputDigitPlace = -1;
	    
	    while (true) {
	        long u = 3L * (3*i+1) * (3*i+2);
	        BigInteger yNum = q.multiply(BigInteger.valueOf(27L*i-12))
	                                .add(r.multiply(FIVE));
	        BigInteger yDen = t.multiply(FIVE);
	        BigInteger y = new BigRational(yNum, yDen).floor();
	        
	        if (y.signum()<0 || y.compareTo(TEN)>=0) {
	            out.format("(%s)", y.toString());
	        } else {
	            out.format("%1s", y.toString());
	        }
	        outputDigitPlace++;
	        if (outputDigitPlace == 0)
	            out.print(".");
	        else if (outputDigitPlace % 20 == 0)
	            out.format(" - %d%n  ", outputDigitPlace);
	        
	        BigInteger qp = q.multiply(BigInteger.valueOf(10L*i*(2*i-1)));
	        BigInteger rp = q.multiply(BigInteger.valueOf(5L*i-2))
	                            .add(r)
	                            .subtract(y.multiply(t))
	                            .multiply(BigInteger.valueOf(10*u));
	        BigInteger tp = t.multiply(BigInteger.valueOf(u));
	        
	        q = qp;
	        r = rp;
	        t = tp;
	        i++;
	    }
	}

	public static void main(String... args) {
	    new GosperPiSpigot().stream();
//		new GosperPiSpigot().fastStream();
	}
}
