package com.bakinsbits.spigot;

import static org.junit.Assert.*;
import org.junit.Test;
import java.lang.Math;

import java.math.BigInteger;

public class BigRationalTest {
	
	// TODO: tests todo: static valueOf methods
	// TODO: tests todo: max, min
	// TODO: no test needed? bitLength, maxBitLength
	// TODO: factor out ValueAndExpected and ValuesAndExpected into generics
	
	final static BigInteger FIVE  = new BigInteger("5");
	final static BigInteger ONE   = BigInteger.ONE;
	final static BigInteger TEN   = BigInteger.TEN;
	final static BigInteger THREE = new BigInteger("3");
	final static BigInteger TWO   = new BigInteger("2");
	final static BigInteger ZERO  = BigInteger.ZERO;

	@Test
	public void testAbs() {
		BigRational zero = new BigRational("0");
		BigRational five = new BigRational("5");
		BigRational mfive = new BigRational("-5");
		assertEquals("both zero", zero, zero.abs());
		assertEquals("both pos", five, five.abs());
		assertEquals("diff sign", five, mfive.abs());		
	}
	
	@Test
	public void testAdd() {
		class ValuesAndExpected {
			final String a;
			final String b;
			final String e;
			ValuesAndExpected(String a, String b, String e) {
				this.a = a;
				this.b = b;
				this.e = e;
			}
		}
		
		ValuesAndExpected[] trials = new ValuesAndExpected[]{
				new ValuesAndExpected("0", "0", "0"),
				new ValuesAndExpected("0", "10", "10"),
				new ValuesAndExpected("10", "1", "11"),
				new ValuesAndExpected("10", "-20", "-10"),
				new ValuesAndExpected("1/2", "1/2", "1"),
				new ValuesAndExpected("1/2", "1/3", "5/6"),
				new ValuesAndExpected("1/3", "1/3", "2/3"),
				new ValuesAndExpected("1/3", "4/3", "5/3"),
				new ValuesAndExpected("2/3", "3", "11/3"),
				new ValuesAndExpected("2/5", "-2/5", "0"),
				new ValuesAndExpected("4/5", "-2/5", "2/5"),
				new ValuesAndExpected("4/3", "5/3", "3")
		};
		
		for (ValuesAndExpected ve : trials) {
			BigRational a = new BigRational(ve.a);
			BigRational b = new BigRational(ve.b);
			BigRational r1 = a.add(b);
			BigRational r2 = b.add(a); // commutative
			assertEquals(ve.a + " + " + ve.b + " -> " + ve.e, ve.e, r1.toString());
			assertEquals(ve.b + " + " + ve.a + " -> " + ve.e, ve.e, r2.toString());
		}
	}
	
	@Test
	public void testApproximateValueOf() {
		
		class Value {
			final long den;
			final long num;
			Value(long num, long den) {
				this.num = num;
				this.den = den;
			}
		}
		
		Value[] trials = new Value[]{
			new Value(10,     1),
			new Value( 0,     1),
			new Value( 1,     1),
			new Value( 1,    10),
			new Value( 1,     2),
			new Value( 1,     5),
			new Value( 1, 10000)
		};
		
		for (Value val : trials) {
			BigRational v = BigRational.valueOf(val.num, val.den);
			double e = (double)val.num / (double)val.den;
			assertEquals(val.toString() + " -> " + e, e, v.approximateValueOf(), Math.ulp(e));
		}
	}
	
	@SuppressWarnings("unused")
	@Test
	public void testBigRationalC1() { // constructor
		BigRational r2 = new BigRational(ZERO);
		BigRational r3 = new BigRational(ONE);
		BigRational r5 = new BigRational(FIVE);
		assertTrue("valid BigRational constructors", true);
	}
	
	@SuppressWarnings("unused")
	@Test
	public void testBigRationalC2() { // constructor
		BigRational r2 = new BigRational(ONE, ONE);
		BigRational r3 = new BigRational(ZERO, ONE);
		BigRational r4 = new BigRational(ONE, FIVE);
		BigRational r5 = new BigRational(ZERO, FIVE);
		assertTrue("valid BigRational constructors", true);
	}
	
	@Test(expected = NullPointerException.class)
	public void testBigRationalF1() { // constructor fails
		@SuppressWarnings("unused")
		BigRational r1 = new BigRational(null, ONE);
		fail("BigRational constructor with numerator null");
	}
	
	@Test(expected = NullPointerException.class)
	public void testBigRationalF2() { // constructor fails
		@SuppressWarnings("unused")
		BigRational r1 = new BigRational(ONE, null);
		fail("BigRational constructor with denominator null");
	}

	@Test(expected = IllegalArgumentException.class)
	public void testBigRationalF3() { // constructor fails
		@SuppressWarnings("unused")
		BigRational r1 = new BigRational(ONE, ZERO);
		fail("BigRational constructor with denominator zero");
	}
	
	@Test
	public void testBigRationalS1() { // constructor, String
		class ValueAndExpected {
			final String e;
			final String v;
			ValueAndExpected(String v, String e) {
				this.v = v;
				this.e = e;
			}
		}
		
		ValueAndExpected[] trials = new ValueAndExpected[]{
				new ValueAndExpected("0", "0"),
				new ValueAndExpected("1", "1"),
				new ValueAndExpected("10", "10"),
				new ValueAndExpected("0/5", "0"),
				new ValueAndExpected("1/1", "1"),
				new ValueAndExpected("5/1", "5"),
				new ValueAndExpected("5/3", "5/3"),
				new ValueAndExpected("3/5", "3/5"),
				new ValueAndExpected("3/6", "1/2"),
				new ValueAndExpected("6/3", "2"),
				new ValueAndExpected("-1", "-1"),
				new ValueAndExpected("-0", "0"),
				new ValueAndExpected("-2/3", "-2/3"),
				new ValueAndExpected("2/-3", "-2/3")
		};
		
		for (ValueAndExpected ve : trials) {
			// Tests both constructor and toString
			BigRational rat = new BigRational(ve.v);
			assertEquals(ve.v + " -> " + ve.e, ve.e, rat.toString());
		}
	}

	@Test
	public void testBigRationalS2() { // constructor, String, radix
		class ValueAndExpected {
			final String e;
			final int radix;
			final String v;
			ValueAndExpected(String v, String e, int radix) {
				this.v = v;
				this.e = e;
				this.radix = radix;
			}
		}
		
		ValueAndExpected[] trials = new ValueAndExpected[]{
				new ValueAndExpected("0", "0", 16),
				new ValueAndExpected("1", "1", 16),
				new ValueAndExpected("10", "10", 16),
				new ValueAndExpected("0/5", "0", 16),
				new ValueAndExpected("1/1", "1", 16),
				new ValueAndExpected("5/1", "5", 16),
				new ValueAndExpected("5/3", "5/3", 16),
				new ValueAndExpected("3/5", "3/5", 16),
				new ValueAndExpected("3/6", "1/2", 16),
				new ValueAndExpected("6/3", "2", 16),
				new ValueAndExpected("-1", "-1", 16),
				new ValueAndExpected("-0", "0", 16),
				new ValueAndExpected("-2/3", "-2/3", 16),
				new ValueAndExpected("2/-3", "-2/3", 16),
				new ValueAndExpected("10/2", "8", 16),
				new ValueAndExpected("14/2", "a", 16),
				new ValueAndExpected("abcd/feb1", "abcd/feb1", 16),
				new ValueAndExpected("abc0/10", "abc", 16),
				new ValueAndExpected("1e/2", "f", 16)
		};
		
		for (ValueAndExpected ve : trials) {
			// Tests both constructor and toString
			BigRational rat = new BigRational(ve.v, ve.radix);
			assertEquals(ve.v + " -> " + ve.e, ve.e, rat.toString(ve.radix));
		}
	}

	@Test
	public void testCompareTo() {
		class ValuesAndExpected {
			final String a;
			final String b;
			final int e;
			ValuesAndExpected(String a, String b, int e) {
				this.a = a;
				this.b = b;
				this.e = e;
			}
		}
		
		ValuesAndExpected[] trials = new ValuesAndExpected[]{
				new ValuesAndExpected("0", "0", 0),
				new ValuesAndExpected("1", "1", 0),
				new ValuesAndExpected("1", "2", -1),
				new ValuesAndExpected("2", "1", 1),
				new ValuesAndExpected("1/2", "1", -1),
				new ValuesAndExpected("4/3", "1", 1),
				new ValuesAndExpected("1/3", "2/3", -1),
				new ValuesAndExpected("1/3", "1/2", -1),
				new ValuesAndExpected("3/5", "3/4", -1)
		};
		
		for (ValuesAndExpected ve : trials) {
			BigRational a = new BigRational(ve.a);
			BigRational b = new BigRational(ve.b);
			assertEquals("compare " + ve.a + " to " + ve.b, ve.e, a.compareTo(b));
		}
	}

	@Test
	public void testDivide() {
		class ValuesAndExpected {
			final String a;
			final String b;
			final String e;
			ValuesAndExpected(String a, String b, String e) {
				this.a = a;
				this.b = b;
				this.e = e;
			}
		}
		
		ValuesAndExpected[] trials = new ValuesAndExpected[]{
				new ValuesAndExpected("10", "1", "10"),
				new ValuesAndExpected("1", "10", "1/10"),
				new ValuesAndExpected("2/3", "1", "2/3"),
				new ValuesAndExpected("1", "2/3", "3/2"),
				new ValuesAndExpected("3/5", "2/5", "3/2"),
				new ValuesAndExpected("3/5", "5/3", "9/25"),
				new ValuesAndExpected("1/2", "1/3", "3/2"),
				new ValuesAndExpected("1/2", "-1/2", "-1"),
				new ValuesAndExpected("-3/7", "-1/2", "6/7")
				// TODO: more test cases?
		};
		
		for (ValuesAndExpected ve : trials) {
			BigRational a = new BigRational(ve.a);
			BigRational b = new BigRational(ve.b);
			BigRational r1 = a.divide(b);
			BigRational r2 = a.multiply(b.reciprocal());
			assertEquals(ve.a + " / " + ve.b + " -> " + ve.e, ve.e, r1.toString());
			assertEquals("confirm multiply via reciprocal", r1, r2);
		}
	}
	
	@Test(expected=ArithmeticException.class)
	public void testDivideF() {
		BigRational r = BigRational.valueOf(10).divide(BigRational.valueOf(0));
		fail("divide by zero failed to throw");
	}

	@Test
	public void testEquals() {
		BigRational oneA = new BigRational(ONE, ONE);
		BigRational oneB = new BigRational(ONE, ONE);
		BigRational oneC = new BigRational(TWO, TWO);
		BigRational oneD = new BigRational(FIVE, FIVE);
		BigRational twoA = new BigRational(TWO, ONE);
		BigRational twoB = new BigRational(TEN, FIVE);
		
		assertEquals("reflexive a", oneA, oneA);
		assertEquals("value type", oneA, oneB);
		assertEquals("reflexive b", oneB, oneA);
		assertEquals("transitive a", oneB, oneC);
		assertEquals("transitive b", oneC, oneD);
		assertEquals("transitive c", oneA, oneD);
		assertEquals("reflexive c", twoA, twoA);
		assertEquals("reflexive d", twoA, twoB);
		assertEquals("reflexive e", twoB, twoA);
		
		assertFalse("different a", oneA.equals(twoA));
		assertFalse("different b", twoA.equals(oneA));
	}

	
	@Test
	public void testFloor() {
		
		class ValueAndExpected {
			final BigInteger e;
			final BigRational v;
			ValueAndExpected(BigRational v, BigInteger e) {
				this.v = v;
				this.e = e;
			}
			ValueAndExpected(long vnum, long vdem, long e) {
				this(BigRational.valueOf(vnum,vdem), BigInteger.valueOf(e));
			}
		}
		
		ValueAndExpected[] trials = new ValueAndExpected[]{
			new ValueAndExpected(  1, 1,   1),
			new ValueAndExpected(- 1, 1,  -1),
			new ValueAndExpected(  4, 1,   4),
			new ValueAndExpected(- 4, 1,  -4),
			new ValueAndExpected(  1, 3,   0),
			new ValueAndExpected(- 1, 3,  -1),
			new ValueAndExpected( 17, 4,   4),
			new ValueAndExpected(-17, 4,  -5),
			new ValueAndExpected(-19, 4,  -5)
		};
				
		for (ValueAndExpected ve : trials) {
			assertEquals(ve.v.toString() + " -> " + ve.e.toString(), ve.e, ve.v.floor());
		}
	}
	
	@Test
	public void testFractionalPart() {
		
		class ValueAndExpected {
			final String e;
			final String v;
			ValueAndExpected(String v, String e) {
				this.v = v;
				this.e = e;
			}
		}
		
		ValueAndExpected[] trials = new ValueAndExpected[]{
				new ValueAndExpected("0", "0"),
				new ValueAndExpected("5", "0"),
				new ValueAndExpected("-5", "0"),
				new ValueAndExpected("1/2", "1/2"),
				new ValueAndExpected("5/2", "1/2"),
  				new ValueAndExpected("-1/2", "1/2"), // note: +1/2
				new ValueAndExpected("-7/4", "1/4")
		};
				
		for (ValueAndExpected ve : trials) {
			BigRational v = new BigRational(ve.v);
			BigRational e = new BigRational(ve.e);
			assertEquals(ve.v + " -> " + ve.e, e, v.fractionalPart());
		}
	}

	@Test
	public void testHashCode() {
		BigRational oneA = new BigRational(ONE, ONE);
		BigRational oneB = new BigRational(ONE, ONE);
		BigRational oneC = new BigRational(TWO, TWO);
		
		assertEquals("value type", oneA.hashCode(), oneB.hashCode());
		assertEquals("transitive a", oneB.hashCode(), oneC.hashCode());
		assertEquals("transitive b", oneA.hashCode(), oneC.hashCode());
	}
	
	@Test
	public void testIsIntegral() {
		assertTrue("is integral 0", BigRational.valueOf(0).isIntegral());
		assertTrue("is integral 4", BigRational.valueOf(4).isIntegral());
		assertTrue("is integral 8", BigRational.valueOf(64, 8).isIntegral());
		assertFalse("is integral 2/3", BigRational.valueOf(2, 3).isIntegral());
		assertTrue("is integral -4", BigRational.valueOf(-8, 2).isIntegral());
	}

	@Test
	public void testIsZero() {
		assertTrue("is zero 0", BigRational.valueOf(0).isZero());
		assertFalse("is zero 1", BigRational.valueOf(1).isZero());
		assertFalse("is zero 2/3", BigRational.valueOf(2,3).isZero());
		assertFalse("is zero -2/3", BigRational.valueOf(-2,3).isZero());
	}
	
	@Test
	public void testMultiply() {
		class ValuesAndExpected {
			final String a;
			final String b;
			final String e;
			ValuesAndExpected(String a, String b, String e) {
				this.a = a;
				this.b = b;
				this.e = e;
			}
		}
		
		ValuesAndExpected[] trials = new ValuesAndExpected[]{
				new ValuesAndExpected("0", "0", "0"),
				new ValuesAndExpected("10", "0", "0"),
				new ValuesAndExpected("5/3", "0", "0"),
				new ValuesAndExpected("10", "1", "10"),
				new ValuesAndExpected("2/3", "1", "2/3"),
				new ValuesAndExpected("3/5", "2/5", "6/25"),
				new ValuesAndExpected("3/5", "5/3", "1"),
				new ValuesAndExpected("1/2", "1/3", "1/6"),
				new ValuesAndExpected("1/2", "-1/2", "-1/4"),
				new ValuesAndExpected("-3/7", "-1/2", "3/14")
				// TODO: more test cases?
		};
		
		for (ValuesAndExpected ve : trials) {
			BigRational a = new BigRational(ve.a);
			BigRational b = new BigRational(ve.b);
			BigRational r1 = a.multiply(b);
			BigRational r2 = b.multiply(a);
			assertEquals(ve.a + " * " + ve.b + " -> " + ve.e, ve.e, r1.toString());
			assertEquals(ve.b + " * " + ve.a + " -> " + ve.e, ve.e, r2.toString());
		}
	}
	
	@Test
	public void testNegate() {
		BigRational zero = BigRational.valueOf(0);
		BigRational five = BigRational.valueOf(5);
		BigRational mfive = BigRational.valueOf(-5);
		assertEquals("both zero", zero, zero.negate());
		assertEquals("five", mfive, five.negate());
		assertEquals("dual", five, five.negate().negate());
		assertEquals("abs", five, five.negate().abs());
	}
	
	@Test
	public void testReciprocal() {
		BigRational one = BigRational.valueOf(1);
		BigRational two = BigRational.valueOf(2);
		BigRational half = BigRational.valueOf(1,2);
		assertEquals("reciprocal 1", one, one.reciprocal());
		assertEquals("reciprocal 2", half, two.reciprocal());
		assertEquals("reciprocal 1/2", two, half.reciprocal());
		assertEquals("reciprocal 2/3", BigRational.valueOf(3,2), BigRational.valueOf(2,3).reciprocal());
		assertEquals("reciprocal 3/2", BigRational.valueOf(2,3), BigRational.valueOf(3,2).reciprocal());
	}
	
	@Test(expected = ArithmeticException.class)
	public void testReciprocalF() {
		@SuppressWarnings("unused")
		BigRational r1 = new BigRational(ZERO, FIVE);
		BigRational r2 = r1.reciprocal();
		fail("BigRational reciprocal of zero");
	}
	
	@Test
	public void testReduce() {
		assertEquals("10/2 - 5/1", "5", BigRational.valueOf(10, 2).toString());
		assertEquals("20/4 - 5/1", "5", BigRational.valueOf(20, 4).toString());
		assertEquals("4/20 - 1/5", "1/5", BigRational.valueOf(4, 20).toString());
		assertEquals("-4/20 - -1/5", "-1/5", BigRational.valueOf(-4, 20).toString());
	}



	@Test
	public void testSignum() {
		assertEquals("signum 0", 0, BigRational.valueOf(0).signum());
		assertEquals("signum 1", 1, BigRational.valueOf(1, 3).signum());
		assertEquals("signum -1", -1, BigRational.valueOf(-3, 2).signum());
	}

	@Test
	public void testSubtract() {
		class ValuesAndExpected {
			final String a;
			final String b;
			final String e;
			ValuesAndExpected(String a, String b, String e) {
				this.a = a;
				this.b = b;
				this.e = e;
			}
		}
		
		ValuesAndExpected[] trials = new ValuesAndExpected[]{
				new ValuesAndExpected("0", "0", "0"),
				new ValuesAndExpected("0", "10", "-10"),
				new ValuesAndExpected("10", "1", "9"),
				new ValuesAndExpected("10", "-20", "30"),
				new ValuesAndExpected("10", "20", "-10"),
				new ValuesAndExpected("1/2", "1/2", "0"),
				new ValuesAndExpected("3/2", "1/2", "1"),
				new ValuesAndExpected("1/2", "1/3", "1/6"),
				new ValuesAndExpected("1/3", "1/3", "0"),
				new ValuesAndExpected("1/3", "5/3", "-4/3"),
				new ValuesAndExpected("2/3", "3", "-7/3"),
				new ValuesAndExpected("2/5", "-2/5", "4/5"),
				new ValuesAndExpected("4/5", "-2/5", "6/5"),
				new ValuesAndExpected("4/3", "5/3", "-1/3"),
				new ValuesAndExpected("5/3", "2/3", "1")
		};
		
		for (ValuesAndExpected ve : trials) {
			BigRational a = new BigRational(ve.a);
			BigRational b = new BigRational(ve.b);
			BigRational r1 = a.subtract(b);
			BigRational r2 = b.subtract(a); // commutative
			assertEquals(ve.a + " - " + ve.b + " -> " + ve.e, ve.e, r1.toString());
			assertEquals(ve.b + " - " + ve.a + " -> -(" + ve.e + ")", ve.e, r2.negate().toString());
		}
	}

	@Test
	public void testToString() {
		assertEquals("zero toString", "0", BigRational.valueOf(0).toString());
		assertEquals("+1 toString", "1", BigRational.valueOf(1).toString());
		assertEquals("-1 toString", "-1", BigRational.valueOf(-1).toString());
		assertEquals("5 toString a", "5", BigRational.valueOf(5).toString());
		assertEquals("5 toString b", "5", BigRational.valueOf(5,1).toString());
		assertEquals("5 toString c", "5", BigRational.valueOf(10,2).toString());
		assertEquals("-5 toString", "-5", BigRational.valueOf(-5).toString());
		assertEquals("5/2 toString", "5/2", BigRational.valueOf(5,2).toString());
		assertEquals("-5/2 toString", "-5/2", BigRational.valueOf(-5,2).toString());
		assertEquals("5/-2 toString", "-5/2", BigRational.valueOf(5,-2).toString());
	}
	
	@Test
	public void testToStringWithRadix() {
		assertEquals("zero toString 16", "0", BigRational.valueOf(0).toString(16));
		assertEquals("1 toString 16", "1", BigRational.valueOf(1).toString(16));
		assertEquals("-1 toString 16", "-1", BigRational.valueOf(-1).toString(16));
		assertEquals("10 toString 16", "a", BigRational.valueOf(10).toString(16));
		assertEquals("16 toString 16", "10", BigRational.valueOf(16).toString(16));
		assertEquals("-30 toString 16", "-1e", BigRational.valueOf(-30).toString(16));
		assertEquals("1/10 toString 16", "1/a", BigRational.valueOf(1,10).toString(16));
		assertEquals("1/16 toString 16", "1/10", BigRational.valueOf(1,16).toString(16));
		assertEquals("-1/10 toString 16", "-1/a", BigRational.valueOf(-1,10).toString(16));
		assertEquals("-1/16 toString 16", "-1/10", BigRational.valueOf(-1,16).toString(16));
	}

}
