package com.bakinsbits.spigot;

import java.io.Serializable;
import java.lang.Math;
import java.lang.Number;
import java.math.BigInteger;

// TODO: constructor from a decimal String, e.g., "10.45"?
// TODO: some kind of logging of bitsizes used

/**
 * @author davidbak@gmail.com
 * 
 *         BigRational: For, you know, really big rational numbers. Like
 *         BigInteger, except for fractions.
 *         
 *         This objects of this class are immutable.
 */
final public class BigRational extends Number implements
		Comparable<BigRational>, Serializable {

	private static final long serialVersionUID = 1L;

	/**
	 * Returns a BigRational whose value is equal to that of the specified long
	 */
	public static BigRational valueOf(long val) {
		return new BigRational(BigInteger.valueOf(val));
	}

	public static BigRational valueOf(long numerator, long denominator) {
		return new BigRational(BigInteger.valueOf(numerator),
				BigInteger.valueOf(denominator));
	}

	private BigInteger numerator, denominator;

	/**
	 * Construct an integral BigRational from a BigInteger
	 * 
	 * @param numerator
	 *            This will be the value of the constructed BigRational
	 */
	public BigRational(BigInteger numerator) {
		if (numerator == null)
			throw new NullPointerException();
		subConstructor(numerator, BigInteger.ONE);
	}

	/**
	 * Construct a rational from two BigIntegers, it will have the value
	 * numerator / denominator
	 * 
	 * @param numerator
	 *            Numerator of constructed BigRational
	 * @param denominator
	 *            Denominator of constructed BigRational
	 */
	public BigRational(BigInteger numerator, BigInteger denominator) {
		if (numerator == null || denominator == null)
			throw new NullPointerException();
		if (denominator.equals(BigInteger.ZERO)) {
			throw new IllegalArgumentException();
		}
		subConstructor(numerator, denominator);
	}

	/**
	 * Construct a rational from a decimal string representation
	 */
	public BigRational(String val) {
		this(val, 10);
	}
	
	/**
	 * Construct a rational from a arbitrary radix string representation, which
	 * is either a) a BigInteger representation, or b) two BigInteger
	 * representations separated by a '/'.
	 */
	public BigRational(String val, int radix) {
		if (val == null)
			throw new NullPointerException();

		String[] parts = val.split("/", -1);
		if (parts.length < 1 || parts.length > 2)
			throw new NumberFormatException();

		BigInteger numerator = new BigInteger(parts[0], radix);
		BigInteger denominator = parts.length == 2 ? new BigInteger(parts[1],
				radix) : BigInteger.ONE;
		subConstructor(numerator, denominator);
	}

	/**
	 * Returns a BigRational whose value is the absolute value of this
	 * BigRational.
	 */
	public BigRational abs() {
		if (numerator.signum() >= 0)
			return this;
		else
			return new BigRational(numerator.abs(), denominator);
	}

	/**
	 * Returns a BigRational whose value is (this + val)
	 * 
	 * @param val
	 *            The second argument to the addition
	 */
	public BigRational add(BigRational val) {
		if (denominator.equals(val.denominator))
			return new BigRational(numerator.add(val.numerator), denominator);
		else
			return new BigRational(numerator.multiply(val.denominator).add(
				denominator.multiply(val.numerator)),
				denominator.multiply(val.denominator));
	}

	/**
	 * Returns a double that approximates the value of this BigRational
	 * 
	 * @return A double that approximates the value of this BigRational
	 */
	double approximateValueOf() {
		return isIntegral() 
		    ? numerator.doubleValue() 
		    : numerator.doubleValue() / denominator.doubleValue();
	}
	
	/**
	 * Checks the invariants for this class. Though, since this is a value
	 * class, shouldn't be needed except at the end of the constructor.
	 */
	private void assertInvariant() {
		assert numerator != null;
		assert denominator != null;
		assert denominator.signum() == 1;
		assert numerator.equals(BigInteger.ZERO) ? denominator.equals(BigInteger.ONE) : true;
				
		// Require that if zero then numerator really is BigInteger.ZERO.  And,
	    // that if integral, denominator really is BigInteger.ONE.
		assert numerator.equals(BigInteger.ZERO) ? numerator == BigInteger.ZERO : true;
		assert denominator.equals(BigInteger.ONE) ? denominator == BigInteger.ONE : true; 
	}

	/**
	 * Returns the bitLength of the numerator and denominator of this
	 * BigRational
	 * 
	 * @return Array of length two, representing the big lengths of the
	 *         numerator and denominator, respectively
	 */
	public int[] bitLength() {
		return new int[] { numerator.bitLength(), denominator.bitLength() };
	}

	/**
	 * Converts this BigRational to a byte
	 */
	@Override
	public byte byteValue() {
		if (isIntegral())
			return numerator.byteValue();
		else
			return (byte) doubleValue();
	}
	
	/**
	 * Compares this BigRational with the specified BigRational
	 */
	public int compareTo(BigRational val) {
		// Easy case is that both denominators are the same (which includes
		// both being integral).
		if (val == null)
			throw new NullPointerException();
		this.assertInvariant();
		val.assertInvariant();
		if (denominator.equals(val.denominator))
			return numerator.compareTo(val.numerator);

		// Now, compare numerators after conversion to common denominator
		return numerator.multiply(val.denominator)
		                .compareTo(val.numerator.multiply(denominator));
	}

	/**
	 * Returns a BigRational whose value is (this / val)
	 */
	public BigRational divide(BigRational val) {
	    // following creates an immediately garbage BigRational, but ok
	    return multiply(val.reciprocal());
	}

	/**
	 * Converts this BigRational to a close double
	 */
	@Override
	public double doubleValue() {
		return approximateValueOf();
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)  // quick shortcut
			return true;
		if (obj == null)  // quick shortcut
			return false;
		// (It would be nice to be able to compare BigInteger to BigRational but
		// this isn't C++)
		if (!(obj instanceof BigRational)) // quick shortcut
			return false;
		
		BigRational that = (BigRational) obj;
		// Works because all BigRationals are normalized and reduced
		this.assertInvariant();
		that.assertInvariant();
		if (this.isIntegral() && that.isIntegral())
			return this.numerator.equals(that.numerator);
		else
			return this.numerator.equals(that.numerator)
			    && this.denominator.equals(that.denominator);
	}

	/**
	 * Converts this BigRational to a close float
	 */
	@Override
	public float floatValue() {
		return (float) doubleValue();
	}

	/**
	 * Returns the largest integer less than or equal to this Rational
	 */
	public BigInteger floor() {
		if (isIntegral())
			return numerator;

		// non-integral: the division truncates, i.e., rounds towards zero
		BigInteger r = numerator.divide(denominator);
		// but for floor we want round towards negative infinity
		if (this.signum() < 0)
			r = r.subtract(BigInteger.ONE);
		return r;
	}

	/**
	 * Gives fractional part of this BigRational
	 * Using the "graham" definition: x - floor(x)
	 * There is an alternate definition, used by Mathematica, se
	 * "Fractional Part" on Wolfram MathWorld.
	 */
	public BigRational fractionalPart() {
		BigRational r = this.subtract(new BigRational(this.floor()));
		return r;
	}

	public BigInteger getDenominator() {
		return denominator;
	}

	public BigInteger getNumerator() {
		return numerator;
	}

	@Override
	public int hashCode() {
		// Works because all BigRationals are normalized and reduced
		assertInvariant();
		return numerator.hashCode() ^ denominator.hashCode();
	}

	/**
	 * Converts this BigRational to an int
	 */
	@Override
	public int intValue() {
		if (isIntegral())
			return numerator.intValue();
		else
			return (int) doubleValue();
	}

	/**
	 * Returns whether or not this BigRational is integral
	 */
	public boolean isIntegral() {
		return denominator == BigInteger.ONE;
	}
	
	/**
	 * Returns whether or not this BigRational is zero
	 */
	public boolean isZero() {
		return numerator == BigInteger.ZERO;
	}

	/**
	 * Converts this BigRational to a long (truncates, doesn't round)
	 */
	public long longValue() {
		if (isIntegral())
			return numerator.longValue();
		else {
			// For int we can convert first to double - a double has enough
			// mantissa bits that we can easily get an int out of it without
			// loss of precision. But this doesn't work for long.
			boolean isPositive = numerator.signum() >= 0;
			BigInteger v = isPositive ? numerator.divide(denominator)
					: numerator.abs().divide(denominator);
			long lv = v.longValue();
			return isPositive ? lv : -lv;
		}
	}

	/**
	 * Returns the maximum of this BigRational and val
	 */
	public BigRational max(BigRational val) {
		return compareTo(val) >= 0 ? this : val;
	}

	/**
	 * Returns the maximum of the bitLength of numerator and denominator
	 */
	public int maxBitLength() {
		return Math.max(numerator.bitLength(), denominator.bitLength());
	}

	/**
	 * Returns the minimum of this BigRational and val
	 */
	public BigRational min(BigRational val) {
		return compareTo(val) <= 0 ? this : val;
	}

	/**
	 * Returns a BigRational whose value is (this * val)
	 */
	public BigRational multiply(BigRational val) {
		if (isIntegral() && val.isIntegral())
			return new BigRational(numerator.multiply(val.numerator), BigInteger.ONE);
		else
			return new BigRational(numerator.multiply(val.numerator),
				                   denominator.multiply(val.denominator));
	}

	/*
	 * Returns a BigRationals of the opposite sign of this one
	 */
	public BigRational negate() {
		if (isZero())
			return this;
		else
			return new BigRational(numerator.negate(), denominator);
	}

	/**
	 * Produce a canonical representation of a rational, so that subsequent
	 * operations are easier. Thus, make sure that 0 is represented as 0/1, and
	 * that the denominator is always positive.
	 */
	private void normalize() {
		if (numerator.equals(BigInteger.ZERO)) {
			numerator = BigInteger.ZERO;
			denominator = BigInteger.ONE;
			return;
		}
		
		boolean isPositive = numerator.signum() == denominator.signum();
		numerator = numerator.abs();
		denominator = denominator.abs();
		if (!isPositive) {
			numerator = numerator.negate();  // if numerator already < 0, then causes a garbage BigInteger
		}
		
		if (denominator.equals(BigInteger.ONE)){
			denominator = BigInteger.ONE;
		}
	}

	/**
	 * Returns 1/this
	 */
	public BigRational reciprocal() {
		if (isZero())
			throw new ArithmeticException("reciprocal of zero");
		return new BigRational(denominator, numerator);
	}

	/**
	 * Reduce the rational to lowest terms.
	 */
	private void reduce() {
		BigInteger gcd = numerator.gcd(denominator);
		if (!gcd.equals(BigInteger.ONE)) // by definition, gcd() is
						     			 // always positive
		{
			numerator = numerator.divide(gcd);
			denominator = denominator.divide(gcd);
		}
	}

	/**
	 * Converts this BigRational to a short
	 */
	@Override
	public short shortValue() {
		if (isIntegral())
			return numerator.shortValue();
		else
			return (short) doubleValue();
	}

	/**
	 * Returns the signum function of this BigRational
	 */
	public int signum() {
		// Works because BigRational is always normalized
		assertInvariant();
		return numerator.signum();
	}

	/**
	 * Helper for multiple constructors
	 */
	private void subConstructor(BigInteger numerator, BigInteger denominator) {
		this.numerator = numerator;
		this.denominator = denominator;

		reduce();
		normalize();
		assertInvariant();
	}

	/**
	 * Returns a BigRational whose value is (this - val)
	 */
	public BigRational subtract(BigRational val) {
		if (denominator.equals(val.denominator))
			return new BigRational(numerator.subtract(val.numerator), denominator);
		else
			return new BigRational(numerator.multiply(val.denominator).subtract(
				denominator.multiply(val.numerator)),
				denominator.multiply(val.denominator));
	}

	/**
	 * Returns the decimal String representation of this BigRational
	 */
	@Override
	public String toString() {
		return toString(10);
	}

	/**
	 * Returns the String representation of this BigRational in the given radix
	 * (as per BigInteger.toString(radix) does not include any radix mark)
	 */
	public String toString(int radix) {
		assertInvariant();
		String s;
//		if (isIntegral()) {
//			s = numerator.toString(radix);
//		} else {
			String num = numerator.toString(radix);
			String den = denominator.toString(radix);

			s = new StringBuilder(num.length() + den.length() + 1)
				.append(num)
				.append(" % ")
				.append(den)
				.toString();
//		}
		return s;
	}

}
